/**
* \file: device_fsm.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "device_fsm.h"

#include <stdio.h>
#include <stdlib.h>

#include "utils/logger.h"
#include "model/device.h"
#include "model/device_list.h"
#include "model/partition_list.h"
#include "control/partition_fsm.h"

static void device_fsm_enter_device_automounted(device_t *device);
static error_code_t device_fsm_enter_device_unmounting(device_t *device,
        device_state_change_callback_t callback_func, void *callback_data);
static void device_fsm_enter_device_unmounted(device_t *device);
static void device_fsm_enter_device_invalid(device_t *device);

static void device_dump_fsm_err(device_state_t coming_from, 
        device_state_t going_to);

//--------------------------------------- API members ------------------------------------
void device_fsm_enter_device_automounting(device_t *device,
        int expected_partition_cnt)
{
    device_state_t coming_from;
    coming_from=device_get_state(device);

    if (coming_from==DEVICE_DETECTED || coming_from==DEVICE_NOMEDIA)
    {
        device_set_state_automounting(device,expected_partition_cnt);
        logger_log_debug("DEVICE_FSM - Device %s in state automounting.",device_get_id(device));
    }
    else
        device_dump_fsm_err(coming_from,DEVICE_AUTOMOUNTING);

}

void device_fsm_enter_device_nomedia(device_t *device)
{
    device_set_state_nomedia(device);
    logger_log_debug("DEVICE_FSM - Device %s in state nomedia",device_get_id(device));
}

void device_fsm_signal_media_removed(device_t *device)
{
    partition_t *partition;

    //at this point, some partitions have not been informed about that
    //they have been removed (probably all). So we are triggering the
    //remove event for them from here. The signal is working synchronous
    //so that we can sure that all partitions have been removed from the
    //model after we are leaving this loop.
    partition=device_first_partition(device,NULL);
    while(partition!=NULL)
    {
        partition_fsm_signal_part_removed(partition);
        //we cannot work with an iterator here since the iterator will get
        //corrupt when a partition is removed from the list
        partition=device_first_partition(device,NULL);
    }

    device_fsm_enter_device_nomedia(device);
}

void device_fsm_signal_device_removed(device_t *device)
{
    partition_t *partition;

    //at this point, some partitions have not been informed about that
    //they have been removed (probably all). So we are triggering the
    //remove event for them from here. The signal is working synchronous
    //so that we can sure that all partitions have been removed from the
    //model after we are leaving this loop.
    partition=device_first_partition(device,NULL);
    while(partition!=NULL)
    {
        partition_fsm_signal_part_removed(partition);
        //we cannot work with an iterator here since the iterator will get
        //corrupt when a partition is removed from the list
        partition=device_first_partition(device,NULL);
    }

    device_fsm_enter_device_invalid(device);
}

error_code_t device_fsm_signal_device_unmount_request(device_t *device,
        device_state_change_callback_t callback_func, void *callback_data)
{
    error_code_t result=RESULT_OK;
    device_state_t coming_from;
    coming_from=device_get_state(device);

    if (coming_from==DEVICE_AUTOMOUNTED)
    {
        result=device_fsm_enter_device_unmounting(device, callback_func, callback_data);
    }
    else
        result=RESULT_NOT_MOUNTED;

    return result;
}

void device_fsm_signal_part_handled(partition_t *partition)
{
    device_t *device;
    device_state_t dev_state;

    device=partition_get_device(partition);
    dev_state=device_get_state(device);

    //we are only interested in this signal when we are in state
    //DEVICE_AUTOMOUNTING or DEVICE_UNMOUNTING
    if (dev_state==DEVICE_AUTOMOUNTING)
    {
        if (device_get_detected_partition_cnt(device) >=
                device_get_expected_partition_cnt(device) &&
                device_get_working_partitions_cnt(device)==0)
            device_fsm_enter_device_automounted(device);
    }

    if (dev_state==DEVICE_UNMOUNTING)
    {
        //nothing is working anymore
        if (device_get_working_partitions_cnt(device)==0)
        {
            //nothing mounted anymore: state->unmounted else state->automounted
            if (device_get_mounted_partitions_cnt(device)==0)
                device_fsm_enter_device_unmounted(device);
            else
                device_fsm_enter_device_automounted(device);
        }
    }
}
//----------------------------------------------------------------------------------------


//--------------------------------------- internal members -------------------------------
static void device_fsm_enter_device_automounted(device_t *device)
{
    device_state_t coming_from;
    coming_from=device_get_state(device);

    if (coming_from==DEVICE_AUTOMOUNTING || coming_from==DEVICE_UNMOUNTING)
    {
        device_set_state_automounted(device);
        logger_log_debug("DEVICE_FSM - Device %s: in state automounted",device_get_id(device));
    }
    else
        device_dump_fsm_err(coming_from,DEVICE_AUTOMOUNTED);
}

static error_code_t device_fsm_enter_device_unmounting(device_t *device,
        device_state_change_callback_t callback_func, void *callback_data)
{
    error_code_t result=RESULT_OK;
    partition_iterator_t itr;
    partition_t *partition;

    device_set_state_unmounting(device);
    logger_log_debug("DEVICE_FSM - Device %s in state unmounting",device_get_id(device));

    device_set_state_change_monitor(device,callback_func,callback_data);

    if (device_get_mounted_partitions_cnt(device)!=0)
    {
        //if we have mounted partitions, kick them of ...
        partition=device_first_mounted_partition(device,&itr);
        while(partition!=NULL && result == RESULT_OK)
        {
            result=partition_fsm_signal_unmount_request(partition,NULL,NULL);
            partition=device_next_mounted_partition(&itr);
        }

        if (result!=RESULT_OK)
            device_set_state_automounted(device);
    }
    else
        device_fsm_enter_device_unmounted(device);

    return result;
}

static void device_fsm_enter_device_unmounted(device_t *device)
{
    device_state_t coming_from;
    coming_from=device_get_state(device);

    if (coming_from==DEVICE_UNMOUNTING)
    {
        device_set_state_unmounted(device);
        logger_log_debug("DEVICE_FSM - Device %s in state unmounted",device_get_id(device));
    }
    else
        device_dump_fsm_err(coming_from,DEVICE_UNMOUNTED);
}

static void device_fsm_enter_device_invalid(device_t *device)
{
    device_set_state_invalid(device);
    logger_log_debug("DEVICE_FSM - Device %s in state invalid",device_get_id(device));

    device_list_delete_device(device);
}

static void device_dump_fsm_err(device_state_t coming_from, 
        device_state_t going_to)
{
    logger_log_error("DEVICE_FSM - We detected an error in the device FSM."
            "Requested State change: %s -> %s",
            device_get_state_str(coming_from),device_get_state_str(going_to));
}
//----------------------------------------------------------------------------------------
